<?php

namespace Pramix\XPayment\Controllers;

use App\Customer;
use App\Http\Controllers\Controller;
use App\Http\Helper;
use App\Invoice;
use Pramix\XUser\Models\User;
use Carbon\Carbon;
use Config;
use Doctrine\DBAL\Driver\IBMDB2\DB2Driver;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Session\Store;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Pramix\XCustomer\Models\CustomerModel;
use Pramix\XGeneral\Models\OptionModel;
use Pramix\XInventory\Models\AverageCostModel;
use Pramix\XInventory\Models\Inventory;
use Pramix\XInvoice\Models\InvoiceModel;
use Pramix\XInvoice\Models\InvoicePaymentModel;
use Pramix\XInvoice\Models\InvoiceProductsModel;
use App\ChequeReturnPaymentHasCheque;
use App\ChequeReturnPayment;
use Pramix\XMedia\Models\MediaModel;
use Pramix\XPayment\Models\ChequeModel;
use Pramix\XProduct\Models\ProductDiscountsModel;
use Pramix\XProduct\Models\ProductsModel;
use Pramix\XProduct\Models\StoreLocationsModel;
use Pramix\XUser\Models\Permission;
use Yajra\DataTables\DataTables;
use App\Utils\ReferenceUtil;

class PaymentController extends Controller
{


    public function index($invoice_id = '', Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        $user = Auth::user();
        $role = $user->roles()->first();
    
        if ($invoice_id == '') {
            $payments = InvoicePaymentModel::with('invoice', 'cheque')
                ->when($request->payment_receipt_no, function ($query, $payment_receipt_no) {
                    return $query->where('invoice_payments.payment_code', $payment_receipt_no);
                })
                ->when($request->payment_date, function ($query, $payment_date) {
                    $dates = explode(' - ', $payment_date);
                    $startDate = date('Y-m-d', strtotime($dates[0]));
                    $endDate = date('Y-m-d', strtotime($dates[1]));
                    return $query->whereBetween('invoice_payments.payment_date', [$startDate, $endDate]);
                })
                ->when($request->customer_name, function ($query, $customer_name) {
                    return $query->whereHas('customer', function ($customerQuery) use ($customer_name) {
                        $customerQuery->where(function ($customerCondition) use ($customer_name) {
                            $customerCondition->where('business_name', 'like', '%' . $customer_name . '%')
                                              ->orWhere('company_name', 'like', '%' . $customer_name . '%');
                        });
                    });
                })
                
                ->when($request->payment_status, function ($query, $payment_status) {
                    if ($payment_status == 'COMPLETED') {
                        return $query->where('invoice_payments.status', 1);
                    } elseif ($payment_status == 'CANCELLED') {
                        return $query->where('invoice_payments.cheque_status', 2);
                    }
                })
                ->when($request->cheque_no, function ($query, $cheque_no) {
                    return $query->whereHas('cheque', function ($sub_query) use ($cheque_no) {
                        $sub_query->where('payment_ref_no', $cheque_no);
                    });
                })
                ->when($request->rep_id, function ($query, $rep_id) {
                    return $query->where('rep_id', $rep_id);
                })
                ->when($request->amount, function ($query, $amount) {
                    return $query->where('invoice_payments.payment_amount', $amount);
                });
    
            if (in_array($user->username, ['user35', 'USER30@GMAIL.COM', 'USER33@GMAIL.COM'])) {
                $payments->where('user_type', '35');
            }
    
            $payments = $payments->whereNull('parent_id')->latest()->paginate(100);
        } else {
            $payments = InvoicePaymentModel::where('invoice_id', $invoice_id)
                ->when($request->rep_id, function ($query, $rep_id) {
                    return $query->where('rep_id', $rep_id);
                });
    
            if (in_array($user->username, ['user35', 'USER30@GMAIL.COM', 'USER33@GMAIL.COM'])) {
                $payments->where('user_type', '35');
            }
    
            $payments = $payments->latest()->paginate(100);
        }
    
        if ($role->id != 1 && (Auth::user()->can(['SHOW_OWN_DATA']) || in_array($role->name, ['REPRESENTATIVE', 'REP']))) {
            $roles = User::where('id', $user->id)->get();
        } else {
            $roles = User::role(['REP', 'REPRESENTATIVE'])->get();
        }
    
        return view('xpayment::payments_list', compact('payments', 'roles'));
    }
    
    
    // public function index(Request $request)
    // {
    //     Permission::checkPermission($request, 'MANAGE_PAYMENT');
    //     return view('xpayment::payment_list');
    // }

    public function viewChequePayments(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_CHEQUE_PAYMENT');
        $cheques = ChequeModel::with('invoice_payment');
        $user = Auth::user();
        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $cheques = $cheques->where('user_type', '35');
        }
        $cheques = $cheques->when($request->customer_name, function ($query, $customer_name) {



            $similarCustomers = CustomerModel::where('company_name', 'like', '%' . $customer_name . '%')->get();
            $customerIds = $similarCustomers->pluck('id')->toArray();


            return $query->whereIn('cheque.customer_id', $customerIds);
        })
        ->when($request->cheque_return_ref_no, function ($query, $cheque_return_ref_no) {
            return $query->where('cheque.cheque_rtn_ref_no', $cheque_return_ref_no);
        })
        ->when($request->received_date, function ($query, $received_date) {
            $dates = explode(' - ', $received_date);
            $startDate = date('Y-m-d', strtotime($dates[0]));
            $endDate = date('Y-m-d', strtotime($dates[1]));

            $dateRange = [$startDate, $endDate];
            return $query->whereBetween('cheque.payment_date', $dateRange);
        })
        ->when($request->cheque_date, function ($query, $cheque_date) {
            $dates = explode(' - ', $cheque_date);
            $startDate = date('Y-m-d', strtotime($dates[0]));
            $endDate = date('Y-m-d', strtotime($dates[1]));

            $dateRange = [$startDate, $endDate];
            return $query->whereBetween('cheque.cheque_date', $dateRange);
        })
        ->when($request->invoice_no, function ($query, $invoice_no) {
            $invoice = InvoiceModel::where('invoice_code', $invoice_no)->first();
            $invoice_id = $invoice->id;
            return $query->whereHas('invoice_payment', function ($subquery) use ($invoice_id) {
                $subquery->where('invoice_id',$invoice_id);
            });
        })
        ->when($request->payment_status, function ($query, $payment_status) {

            if ($payment_status == 'Pending') {
                return $query->where('cheque.status', 0);
            }
            if ($payment_status == 'Reject') {
                return $query->where('cheque.status', 2);
            }
            if ($payment_status == 'Cleared') {
                return $query->where('cheque.status', 1);
            }


        })
        ->latest()->paginate(100);
        return view('xpayment::cheque_payments_list', compact('cheques'));
    }
    // public function viewChequePayments(Request $request)
    // {
    //     Permission::checkPermission($request, 'MANAGE_CHEQUE_PAYMENT');
    //     return view('xpayment::cheque_payment_list');
    // }

    public function create(Request $request)
    {
        Permission::checkPermission($request, 'ADD_PAYMENT');

        $customer_id = $request['customer_id'] ?? '';
        $roles = User::role(['REPRESENTATIVE','REP'])->get();
        $page = 'payment';
        return view('xpayment::create_payment', compact('roles'))
            ->with('customer_id', $customer_id)
            ->with('page', $page);
    }


    public function store(Request $request)
    {
        //
    }


    public function show($id)
    {
        //
    }

    public function edit(Request $request, $id)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        $payments = InvoicePaymentModel::with('invoice')->where('id', $id)->first();

        return view('xpayment::edit_payment')
            ->with('customer', $payments->invoice->customer)
            ->with('user', $payments->invoice->user)
            ->with('invoice', $payments->invoice)
            ->with('payment', $payments);
    }

    public function PaymentEdit(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        parse_str($request['payment_details'], $payment_details);

        $validator = Validator::make($payment_details, [
            'payment_method_selected' => 'required',
            'cheque_bank' => 'required_if:payment_method_selected,cheque',
            'payment_amount' => 'required',
        ]);
        if (!$validator->passes()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()->all()]);
        }

        $invoice = InvoiceModel::find($payment_details['invoice_id']);
        $invoice_balance = $invoice->balance;

        $customer = CustomerModel::find($invoice->customer_id);
        $payment = InvoicePaymentModel::find($payment_details['payment_id']);

        $before_amount = $payment->payment_amount;
        $payment_amount = $payment_details['payment_amount'];



        if ($payment_details['payment_method_selected'] == 'cheque') {
            $cheque = ChequeModel::saveCheque($invoice->customer_id, $payment_details['payment_amount'], $payment_details, $payment->cheque_id);
            $payment->cheque_status = 0;
            $payment->bank_id = $payment_details['cheque_bank'];
            $payment->cheque_id = $cheque->id;
        }
        $payment->status = 1;
        $payment->cheque_date = $payment_details['cheque_date'];
        $payment->payment_date = $payment_details['date'];
        $payment->payment_method = $payment_details['payment_method_selected'];
        $payment->payment_ref_no = $payment_details['payment_ref_no'];
        $payment->payment_remarks = $payment_details['payment_remarks'];
        $payment->payment_amount = $payment_details['payment_amount'];
        $payment->save();


        $defence_amount = $before_amount - $payment_amount;
        $invoice->paid_amount = $invoice->paid_amount - $defence_amount;
        $invoice->balance = $invoice->balance + $defence_amount;
        $invoice->save();


        if ($payment_details['payment_method_selected'] == 'credit') {
            if ($customer != NULL) {
                $customer->credit_balance += $defence_amount;
                $customer->save();
            }
        }

        if ($invoice->balance < $invoice_balance) {
            if ($customer != NULL) {
                $credit = $defence_amount;
                if ($defence_amount > 0)
                    $customer->credit_balance -= $credit;
                else
                    $customer->credit_balance += $credit;
                $customer->save();
            }
        }

        if ($customer != NULL) {
            CustomerModel::updateCustomerOutStanding($invoice->customer_id);
        }

        return response()->json(['status' => 'success', 'msg' => __('common.messages.save_successfully')]);
    }



    public function update(Request $request, $id)
    {
        //
    }


    public function destroy(Request $request, $id)
    {
        // dd($id);
        try {
            DB::beginTransaction();
            $payment = InvoicePaymentModel::find($id);
            $payment_status = $payment->status;

            if($payment->payment_method == 'cheque')
            {
                $cheque = ChequeModel::find($payment->cheque_id);

                // get the cheque return payment id
                $cheque_return_payment_cheque = ChequeReturnPaymentHasCheque::where('cheque_id', $cheque->id)->first();


                if($cheque_return_payment_cheque !== null)
                {
                    //get all the cheque return payments
                    $cheque_return_payment_cheques = ChequeReturnPaymentHasCheque::where('cheque_id', $cheque->id)->get();
                    $cheque_return_payment_cheque_id = $cheque_return_payment_cheque->cheque_payment_id;
                    // get cheque return payment
                    $cheque_return_payments = ChequeReturnPayment::where('customer_id', $cheque->customer_id)->get();
                    // delete entries of ChequeReturnPaymentHasCheque
                    if(!empty($cheque_return_payment_cheque->id))
                    {
                        foreach($cheque_return_payment_cheques as $cheque_return_payment_cheque)
                        {
                            $cheque_return_payment_cheque->delete();
                        }

                    }

                    foreach($cheque_return_payments as $cheque_return_payment)
                    {
                        $cheque_return_payment->delete();
                    }
                }
                // delete the cheque
                $cheque->delete();
            }

            // delete child payments
            $child_payments = InvoicePaymentModel::where('parent_id', $payment->id)->get();
            if(!empty($child_payments))
            {
                foreach($child_payments as $child_payment)
                {
                    // Reason for this code is in payment create page the value for the payment display datatable was shown from db
                    // if you delete in payment create datatable data will be wrong

                    if ($payment_status == true || $payment_status == false ) {
                        $invoice = Invoice::find($child_payment->invoice_id);
                        $customer = Customer::find($invoice->customer_id);
                        $payment_amount = $child_payment->payment_amount;
                        $customer->outstanding_amount += $payment_amount;
                        $total = $invoice->total;
                        $returned_amount = $invoice->returned_amount;
                        $paid_amount = $invoice->paid_amount - $payment_amount;
                        $balance = $total - ($returned_amount + $paid_amount);
                        $invoice->paid_amount = $paid_amount;
                        $invoice->balance = $balance;
                        $customer->update();
                        $invoice->update();
                    }
                    $child_payment->delete();
                }
            }
            // delete the payment
            $payment->delete();
            DB::commit();
            return response()->json(['status' => 'success', 'message' => 'Payment delete success.!']);
        } catch (Exception $e) {
            DB::rollBack();
            // dd($e);
            return response()->json(['status' => 'error', 'message' => 'Something went wrong.!']);
        }
    }

    public function saveInvoicePayment(Request $request)
    {
        // dd($request);

        Permission::checkPermission($request, 'ADD_PAYMENT');
        parse_str($request['payment_details'], $payment_details);
        $payment_amount = $payment_details['payment_amount'];
        $selected_invoices = $request['invoices_selected'];

        $validator = Validator::make($payment_details, [
            'payment_method' => 'required',
            'cheque_date' => 'required_if:payment_method,cheque'
        ]);
        if (!$validator->passes()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()->all()]);
        }
        if ($selected_invoices == null) {
            return response()->json(['status' => 'error', 'msg' => 'Please select invoice']);
        }

        asort($selected_invoices);
        if ($payment_details['payment_method'] == 'credit') {
            $customer = CustomerModel::find($request['customer_id']);
            if ($customer->credit_balance < $payment_amount) {
                return response()->json(['status' => 'error', 'msg' => 'Please check the credit balance']);
            }
        }

        if ($payment_details['payment_method'] == 'cheque') {
            $cheque = ChequeModel::saveCheque($request['customer_id'], $payment_amount, $payment_details);
        }

        // create parent payment data



        // if (!empty($payment_details['payment_code'])) {
        //     $payment_code = $payment_details['payment_code'];
        // } else {
        //     $last_record = InvoicePaymentModel::where('user_type', $invoice->user_type)->orderBy('id', 'desc')->first();
        //     $prefix = ($invoice->user_type == 10) ? 'IP' : 'BP';
        //     $payment_code = OptionModel::generateCode($prefix, 4, $last_record->payment_code ?? NULL);
        // }

        // assign payment code
        $pay_customer = CustomerModel::find($request['customer_id']);
        $customer_type = $pay_customer->invoice_type;
        $sub_type = '';
        $prefix = '';
        $payment_code = '';
        if($customer_type == 10)
        {

            $sub_type = 10;
            $prefix = 'IP';
            $payment_code_number = ReferenceUtil::returnReferenceNumber($sub_type,$prefix);
            $payment_code_int = intval($payment_code_number->no) + 1;
            ReferenceUtil::referenceUpdate($payment_code_int,$prefix);
            $payment_code = 'IP-'. str_pad($payment_code_int, 4, '0', STR_PAD_LEFT);
            // dd($payment_code);
        }
        if($customer_type == 35)
        {

            $sub_type = 35;
            $prefix = 'BP';
            $payment_code_number = ReferenceUtil::returnReferenceNumber($sub_type,$prefix);
            $payment_code_int = intval($payment_code_number->no) + 1;
            ReferenceUtil::referenceUpdate($payment_code_int,$prefix);
            $payment_code = 'BP-'. str_pad($payment_code_int, 4, '0', STR_PAD_LEFT);
            // dd($payment_code);
        }


        $payment_customer = CustomerModel::find($request['customer_id']);
        $payment = InvoicePaymentModel::create([
            'payment_code' => $payment_code,
            'invoice_id' => 0,
            'cheque_date' => $payment_details['cheque_date'],
            'payment_date' => $payment_details['payment_date'],
            'payment_method'=> $payment_details['payment_method'],
            'payment_ref_no'=> $payment_details['payment_ref_no'],
            'payment_remarks'=> $payment_details['payment_remarks'],
            'payment_amount' =>$payment_details['payment_amount'],
            'rep_id' => $payment_details['rep'],
            'parent_id'=> NULL,
            'customer_id' =>$request['customer_id'],
            'user_type' =>!empty($payment_customer->invoice_type) ? $payment_customer->invoice_type : '10',

        ]);

        if ($payment_details['payment_method'] == 'cheque') {
            $payment->cheque_status = 1;
            $payment->status = 1;
            $payment->cheque_id = $cheque->id ?? NULL;
            $payment->bank_id = $payment_details['cheque_bank'];
        }
        $payment->save();

        // dd($payment);
        $parent_id = $payment->id;
        $payment_code  = $payment->payment_code;


        foreach ($selected_invoices as $invoice_id) {
            $invoice_payment_amount = $invoice_id['amount'];
            InvoiceModel::saveInvoicePayment($invoice_id['invoice_id'], $invoice_payment_amount, $payment_details, $parent_id,$payment_code);
        }

        if ($payment_amount > 0) {
            $customer = CustomerModel::find($request['customer_id']);
            $customer->credit_balance += $payment_amount;
            $customer->save();
        }

        return response()->json(['status' => 'success', 'msg' => __('common.messages.save_successfully')]);
    }

    public function getInvoiceByCustomer($customer_id = '', Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_INVOICE');
        if ($customer_id == '') {
            $invoice = [];
        } else {
            $invoice = InvoiceModel::where('balance','>', 0)->where('customer_id', $customer_id)->with('customer')->get();
        }
        $edit_invoice_permission = false;
        if (Auth::user()->can('EDIT_INVOICE')) {
            $edit_invoice_permission = true;
        }
        return Datatables::of($invoice)
            ->addColumn('action', function ($invoice) use ($edit_invoice_permission) {
                if ($edit_invoice_permission) {
                    return '<a class = "btn btn-info btn-xs" href="' . url("/invoice/" . $invoice->id . "/edit") . '" id="edit_invoice" data-original-title="" title=""><i class="fa fa-pencil"></i></a> ';
                }
            })
            ->addColumn('created_by', function ($invoice) {
                return $invoice->user->username;
            })
            ->addColumn('customer', function ($invoice) {
                if (isset($invoice->customer->fullname))
                    return $invoice->customer->fullname;
            })
            ->editColumn('payment_status', function ($invoice) {

                if ($invoice->total == $invoice->paid_amount)
                    return 'Completed';
                elseif ($invoice->total == $invoice->balance)
                    return 'Pending';
                elseif ($invoice->total != $invoice->balance)
                    return 'Partial';
            })
            ->editColumn('status', function ($invoice) {
                if ($invoice->status == 'D')
                    return 'Draft';
                elseif ($invoice->status == 'A')
                    return 'Completed';
                elseif ($invoice->status == 'C')
                    return 'Cancelled';
            })
            ->editColumn('total', function ($invoice) {
                return Helper::formatPrice($invoice->total);
            })
            ->editColumn('paid_amount', function ($invoice) {
                $invoice_paid_amount_total = \App\Utils\CommonUtil::calculatePaidAmountInvoice($invoice->id);
                return Helper::formatPrice($invoice_paid_amount_total);
            })
            ->editColumn('balance', function ($invoice) {
                $invoice_balance_total = \App\Utils\CommonUtil::calculateInvoiceOutstanding($invoice->id);
                return Helper::formatPrice($invoice_balance_total);
            })
            ->make(true);
    }

    public function getPaymentList($invoice_id = '', Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        if ($invoice_id == '') {
            $payments = InvoicePaymentModel::with('invoice')->get();
        } else {
            $payments = InvoicePaymentModel::where('invoice_id', $invoice_id)->get();
        }
        $user = Auth::user();
        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $payments = $payments->where('user_type', '35');
        }


        return Datatables::of($payments)
            ->editColumn('action', function ($payments) {
                $html = "<button class='btn btn-warning btn-sm payment-view-button fa fa-eye' aria-hidden='true'></button>
                            <a href=" . url("/payment/" . $payments->id . "/edit") . "><button class='btn btn-blue btn-sm fas fa-edit payment-edit-button'></button></a> <button class='btn btn-danger btn-sm delete_payment fa fa-trash-o' aria-hidden='true'></button>";
                return $html;
            })
            ->editColumn('status', function ($payments) {
                if ($payments->status != 0)
                    return '<span class="text-success"><strong>Completed</strong></span>';
                elseif ($payments->cheque_status == 2) {
                    return '<span class="text-danger"><strong>Cancelled</strong></span>';
                } else
                    return '<span class="text-warning"><strong>Pending</strong></span>';
                // return '<span class="text-danger"><strong>Cancelled</strong></span>';
            })
            ->editColumn('rep_id', function($payments){
                if(!empty($payments->rep_id))
                {
                    $rep = User::find($payments->rep_id);
                    return $rep->username;
                }
                else{
                    return 'N/A';
                }
            })
            ->addColumn('customer', function ($payments) {
                if (isset($payments->invoice->customer->company_name))
                    return $payments->invoice->customer->company_name;
            })
            ->addColumn('invoice_code', function ($payments) {
                $invoice_code = $payments->invoice->invoice_code ?? '';
                return $invoice_code;
            })
            ->addColumn('customer_mobile', function ($payments) {
                return $payments->invoice->customer->mobile ?? '';
            })
            ->editColumn('payment_amount', function ($payment) {
                return number_format($payment->payment_amount, 2);
            })
            ->addColumn('invoice_id', function ($payments) {
                return $payments->invoice->id;
            })
            ->rawColumns(['action', 'status'])
            ->make(true);
    }

    public function getChequePaymentsList(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_CHEQUE_PAYMENT');
        $cheques = ChequeModel::with('invoice_payment')->get();
        $user = Auth::user();
        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $cheques = $cheques->where('user_type', '35');
        }
        $approve_or_reject_cheque_payment_permission = false;
        if (Auth::user()->can('MANAGE_CHEQUE_PAYMENT')) {
            $approve_or_reject_cheque_payment_permission = true;
        }

        return Datatables::of($cheques)
            ->editColumn('action', function ($payments) use ($approve_or_reject_cheque_payment_permission) {
                $approve = '';
                $delete = '';
                if ($approve_or_reject_cheque_payment_permission && $payments->status == 0) {
                    $approve = '<button class="btn btn-success btn-sm payment-approve-button fa fa-check" aria-hidden="true"></button>';
                    $delete = '<button class="btn btn-danger btn-sm payment-delete-button fas fa-times" aria-hidden="true"></button>';
                }
                if ($approve_or_reject_cheque_payment_permission && $payments->status == 1) {
                    $delete = '<button class="btn btn-danger btn-sm payment-delete-button fas fa-times" aria-hidden="true"></button>';
                }
                return $approve . ' ' . $delete;
            })
            ->editColumn('status', function ($payments) {
                if ($payments->status == 0)
                    return '<span class="text-default"><strong>Pending</strong></span>';
                elseif ($payments->status == 2)
                    return '<span class="text-danger"><strong>Reject</strong></span>';
                else
                    return '<span class="text-success"><strong>Cleared</strong></span>';
            })
            ->editColumn('cheque_rtn_ref_no', function ($payment) {
                return $payment->cheque_rtn_ref_no ?? 'N/A';
            })
            ->editColumn('cheque_return_date', function ($payment) {
                return $payment->cheque_return_date ?? 'N/A';
            })
            ->addColumn('customer', function ($payments) {
                return $payments->customer->company_name;
            })
            ->addColumn('invoice', function ($payments) {
                if (!empty($payments->invoice_payment)) {
                    return '<a href="' . url('invoice/' . $payments->invoice_payment->invoice_id) . '/edit" target="_blank">' . $payments->invoice_payment->invoice->invoice_code . '</a>';
                }
                return 'N/A';
            })
            ->addColumn('customer_mobile', function ($payments) {
                return $payments->customer->mobile;
            })

            ->rawColumns(['action', 'status', 'invoice'])
            ->make(true);
    }

    public function rejectPayment(Request $request)
    {
        $cheque_id  = $request['cheque_id'];
        $cheque = ChequeModel::find($cheque_id);
        $cheque->cheque_return_date = now();
        $cheque->status = 2;

        $user_type = $cheque->user_type;
        $prefix = '';
        $sub_type = '';


        if($user_type == 10)
        {
            $sub_type = 10;
            $prefix = 'CQ';
            $cheque_rtn_ref_number = ReferenceUtil::returnReferenceNumber($sub_type,$prefix);
            $cheque_rtn_ref_code_int = intval($cheque_rtn_ref_number->no) + 1;
            ReferenceUtil::referenceUpdate($cheque_rtn_ref_code_int,$prefix);
            $new_ref_no = 'CQ-'. str_pad($cheque_rtn_ref_code_int, 4, '0', STR_PAD_LEFT);

        }
        if($user_type == 35)
        {
            $sub_type = 35;
            $prefix = 'BQ';
            $cheque_rtn_ref_number = ReferenceUtil::returnReferenceNumber($sub_type,$prefix);
            $cheque_rtn_ref_code_int = intval($cheque_rtn_ref_number->no) + 1;
            ReferenceUtil::referenceUpdate($cheque_rtn_ref_code_int,$prefix);
            $new_ref_no = 'BQ-'. str_pad($cheque_rtn_ref_code_int, 4, '0', STR_PAD_LEFT);

        }



        // Update the cheque's cheque_rtn_ref_no
        $cheque->cheque_rtn_ref_no = $new_ref_no;
        $cheque->save();

        $parent_payment = InvoicePaymentModel::where('cheque_id', $cheque->id)->first();
        $parent_payment->status = 0;
        $parent_payment->cheque_status = 2;
        $parent_payment->save();

        $payments = InvoicePaymentModel::where('parent_id', $parent_payment->id)->where('payment_method','cheque')->get();

        foreach ($payments as $payment) {
            InvoiceModel::updateInvoicePayments($payment->invoice_id);
        }

        return response()->json(['status' => 'success', 'msg' => 'Cheque rejected']);
    }

    public function approvePayment(Request $request)
    {
        $cheque_id  = $request['cheque_id'];

        $cheque = ChequeModel::find($cheque_id);
        $cheque->status = 1;
        $cheque->save();

        $payments = InvoicePaymentModel::where('cheque_id', $cheque->id)->get();

        foreach ($payments as $payment) {
            $payment->status = 1;
            $payment->cheque_status = 1;
            $payment->save();
//            InvoiceModel::updateInvoicePayments($payment->invoice_id);
        }


        return response()->json(['status' => 'success', 'msg' => __('common.messages.save_successfully')]);
    }


    public function viewPayment(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        $payments = InvoicePaymentModel::with('invoice')->where('id', $request['id'])->first();
        $customer = $payments->invoice->customer ?? $payments->customer;
        $child_payments = InvoicePaymentModel::where('parent_id', $payments->id)->get();
        return view('xpayment::payment_view', compact('child_payments'))
            ->with('customer', $customer)
            ->with('user', $payments->user)
            // ->with('invoice', $payments->invoice) FOR NOW INVOICES ARE COMMENTED
            ->with('payments', $payments);
    }

    public function calPaymentePrice(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_PAYMENT');
        $vat = false;
        $nbt = false;
        $vat_amount = 0;
        $nbt_amount = 0;
        $invoice_id = $request['invoice_id'];

        if ($invoice_id != null) {

            $paid_amount = InvoicePaymentModel::where('invoice_id', $invoice_id)->where('status', 1)->sum('payment_amount');
            $sub_tot = InvoiceProductsModel::where('invoice_id', $invoice_id)->sum('sub_total');
            $record = InvoiceModel::where('id', $invoice_id)->first();


            if ($record['vat_amount'] != 0) {
                dd($record['vat_amount']);
                $vat = true;
            }
            if ($record['nbt_amount'] != 0) {
                dd($record['nbt_amount']);
                $nbt = true;
            }

            if ($record == NULL)
                $record = new InvoiceProductsModel();

            $discount = $record['discount'];
            $discount_type = $record['discount_type'];;


            if ($discount_type == 'P') {
                $total = $sub_tot * (100 - $discount) / 100;
            } else {
                $total = $sub_tot - $discount;
            }

            if ($vat) {
                $vat_amount = $total * (getConfig('VAT_PERCENTAGE') / 100);
            }
            if ($nbt) {
                $nbt_amount = $total * (getConfig('NBT_PERCENTAGE') / 100);
            }

            $total = $total + $vat_amount + $nbt_amount;
            if ($paid_amount != 0) {
                $balance = $total - $paid_amount;
            } else
                $balance = $total;

            $record->sub_total = $sub_tot;
            $record->discount = $discount;
            $record->discount_type = $discount_type;
            $record->vat_amount = $vat_amount;
            $record->nbt_amount = $nbt_amount;
            $record->total = round($total, 2);
            $record->paid_amount = round($paid_amount, 2);
            $record->balance = round($balance, 2);
            $record->save();

            CustomerModel::updateCustomerOutStanding($record->customer_id);

            return response()->json(['status' => 'success', 'msg' => __('common.messages.save_successfully'), 'record' => $record]);
        } else
            return 0;
    }

    public function updateAllCheques()
    {
        try {
            $current_date = now();
            $cheques = ChequeModel::where('status', 0)->where('cheque_date', '<=', $current_date)->get();
            DB::beginTransaction();
            foreach ($cheques as $cheque) {
                $cheque_date = $cheque->cheque_date;
                $different_in_days = $current_date->diffInDays($cheque_date);
                if ($different_in_days > 7) {
                    $cheque->status = 1;
                    $cheque->update();
                    $payments = InvoicePaymentModel::where('cheque_id', $cheque->id)->get();
                    foreach ($payments as $payment) {
                        $payment->status = 1;
                        $payment->cheque_status = 1;
                        $payment->update();
                        InvoiceModel::updateInvoicePayments($payment->invoice_id);
                    }
                }
            }
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
        }
    }

    public function getSelectedInvoiceBalance(Request $request)
    {
        try {
            $selected_invoices = $request->selected_invoices;
            $invoices_balance = Invoice::whereIn('id', $selected_invoices)->sum('balance');
            return response()->json(number_format($invoices_balance, 2));
        } catch (Exception $e) {
            dd($e);
        }
    }
}
