<?php

namespace Pramix\XCustomer\Controllers;


use App\Http\Controllers\Controller;
use App\Http\Helper;
use App\Models\CustomerAddressesModel;
use App\Notifications\CustomerNotifications;
use Config;
use Countries;
use App\Rules\BranchUniqueValidator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\ToArray;
use Pramix\XCustomer\Models\CustomerModel;
use Pramix\XGenaral\Models\CommentsModel;
use Pramix\XGeneral\Models\AddressModel;
use Pramix\XPayment\Models\ChequeModel;
use Pramix\XUser\Models\Permission;
use Pramix\XUser\Models\User;
use Yajra\DataTables\DataTables;
use App;
use App\Customer;
use App\Exports\CustomersExport;
use App\Utils\CommonUtil;
use Carbon\Carbon;
use Maatwebsite\Excel\Excel as ExcelExcel;
use Maatwebsite\Excel\Facades\Excel;

class CustomerController extends Controller
{

    public function index(Request $request)
    {
        $this->authorize('MANAGE_CUSTOMERS');
    
        $user = Auth::user();
        $role = $user->roles()->first();
    
        $roles = User::role(['REP', 'REPRESENTATIVE'])->get();
    
        $customers = CustomerModel::with('rep');
    
        if ($role->id != 1 && ($user->can(['SHOW_OWN_DATA']) || $role["name"] == 'REPRESENTATIVE' || $role["name"] == 'REP')) {
            $customers = $customers->where('rep_id', $user->id);
        }
    
        $customers = $customers
            ->when($request->business_name, function ($query, $business_name) {
                return $query->where('business_name', $business_name);
            })
            ->when($request->company_name, function ($query, $company_name) {
                return $query->where('company_name', 'like', '%' . $company_name . '%');
            })
            ->when($request->mobile, function ($query, $mobile) {
                return $query->where('mobile', $mobile);
            })
            ->when($request->telephone, function ($query, $telephone) {
                return $query->where('telephone', $telephone);
            })
            ->when($request->email, function ($query, $email) {
                return $query->where('email', $email);
            })
            ->when($request->nic, function ($query, $nic) {
                return $query->where('nic', $nic);
            })
            ->when($request->customer_grade, function ($query, $customer_grade) {
                return $query->where('customer_grade', $customer_grade);
            });
    
        if (in_array($user->username, ['user35', 'USER30@GMAIL.COM', 'USER33@GMAIL.COM'])) {
            $customers->where('invoice_type', '35');
        }
    
        $customers = $customers->latest()->paginate(10);
    
        return view('xcustomer::index', compact('customers', 'roles'));
    }
    
      //COMMENTED BY PURPOSE THIS CODE MIGHT NEED FOR THE FUTURE USE

    // public function index(Request $request)
    // {
    //     Permission::checkPermission($request, 'MANAGE_CUSTOMERS');

    //     $user = Auth::user();

    //     $role = $user->roles()->first();
    //     // dd($role);
    //     if($role->id != 1 && Auth::user()->can(['SHOW_OWN_DATA']) )
    //     {
    //         $roles = User::where('id', $user->id)->get();
    //     }
    //     else if ($role["name"] !== 'REPRESENTATIVE'){
    //         $roles = User::role('REPRESENTATIVE')->get();
    //     }

    //     return view('xcustomer::all_customer_list', compact('roles'));
    // }


    public function create(Request $request)
    {

        Permission::checkPermission($request, 'ADD_CUSTOMER');

        $countryList = json_decode(Countries::getList(App::getLocale(), 'json'));

        $allow_comment = TRUE;
        $page = 'customer';
        return view('xcustomer::create_customer')
            ->with('countryList', $countryList)
            ->with('allow_comment', $allow_comment)
            ->with('page', $page);
    }

    public function store(Request $request)
    {

        Permission::checkPermission($request, 'ADD_CUSTOMER');

        parse_str($request['customer_details'], $customer_details);
        parse_str($request['business_address_details'], $business_address_details);
        parse_str($request['shipping_address_details'], $shipping_address_details);

        if ($request['mobile'] != NULL)
            $customer_details['formated_mobile'] = $request['mobile'];
        if ($request['telephone'] != NULL)
            $customer_details['formated_telephone'] = $request['telephone'];

        if ($request['fax'] != NULL)
            $customer_details['formated_fax'] = $request['fax'];
        if ($customer_details['website'] != '')
            $customer_details['website'] = strpos($customer_details['website'], 'http') !== 0 ? "http://" . $customer_details['website'] . "" : $customer_details['website'];

        $customMessages = [
            'formated_mobile.phone' => 'Invalid Mobile No.',
            'formated_fax.phone' => 'Invalid Fax No.',
            'formated_telephone.phone' => 'Invalid Telephone No.',
        ];
        $validator = Validator::make($customer_details, [
            'business_name' => ['required', new BranchUniqueValidator(new CustomerModel(), 'business_name')],
            'formated_mobile' => ['phone:' . $request['mobile_country'], new BranchUniqueValidator(new CustomerModel(), 'mobile')],
            'formated_telephone' => ['phone:' . $request['telephone_country'], new BranchUniqueValidator(new CustomerModel(), 'telephone')],
            'formated_fax' => ['phone:' . $request['fax_country']],
            'email' => ['email', new BranchUniqueValidator(new CustomerModel(), 'email')],
            'nic' => ['max:20', new BranchUniqueValidator(new CustomerModel(), 'nic')],
            'title' => 'required',
            'website' => 'url',
            'rep' => ['required'],
            //'area' => ['required'],
            'invoice_type' => ['required'],
            'customer_grade' => ['sometimes', 'string'],
        ], $customMessages);

        if (!$validator->passes()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()->all()]);
        }


        try {
            $customer = CustomerModel::storeCustomer($customer_details);
            AddressModel::saveAddresses($business_address_details, 'B', $customer->id, 'C');
            AddressModel::saveAddresses($shipping_address_details, 'S', $customer->id, 'C');
            return response()->json(['status' => 'success', 'msg' => __('common.messages.save_successfully'), 'id' => $customer->id, 'business_name' => $customer->business_name, 'full_name' => $customer->fname . ' ' . $customer->lname]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'msg' => __('common.messages.save_error')]);
        }
    }

    public function show($id)
    {
        //
    }

    public function edit(Request $request, $id)
    {
        Permission::checkPermission($request, 'EDIT_CUSTOMER');

        $customer = CustomerModel::findOrFail($id);
        $countryList = json_decode(Countries::getList(App::getLocale(), 'json'));
        $business_address = AddressModel::where('ref_id', $id)->where('user_type', getConfigArrayValueByKey('ADDRESS_USER_TYPE', 'customer'))->where('address_type', 'B')->first();

        $shipping_address = AddressModel::where('ref_id', $id)->where('user_type', getConfigArrayValueByKey('ADDRESS_USER_TYPE', 'customer'))->where('address_type', 'S')->first();

        return view('xcustomer::create_customer')
            ->with('customer', $customer)
            ->with('shipping_address', $shipping_address)
            ->with('business_address', $business_address)
            ->with('countryList', $countryList);
    }


    public function update(Request $request, $id)
    {

        Permission::checkPermission($request, 'EDIT_CUSTOMER');

        parse_str($request['customer_details'], $customer_details); //This will convert the string to array
        parse_str($request['business_address_details'], $business_address_details);
        parse_str($request['shipping_address_details'], $shipping_address_details);

        if ($request['mobile'] != NULL)
            $customer_details['formated_mobile'] = $request['mobile'];
        if ($request['telephone'] != NULL)
            $customer_details['formated_telephone'] = $request['telephone'];
        if ($request['fax'] != NULL)
            $customer_details['formated_fax'] = $request['fax'];

        if ($customer_details['website'] != '')
            $customer_details['website'] = strpos($customer_details['website'], 'http') !== 0 ? "http://" . $customer_details['website'] . "" : $customer_details['website'];

        $customMessages = [
            'formated_mobile.phone' => 'Invalid Mobile No.',
            'formated_fax.phone' => 'Invalid Fax No.',
            'formated_telephone.phone' => 'Invalid Telephone No.',
        ];

        $validator = Validator::make($customer_details, [
            'business_name' => ['required', new BranchUniqueValidator(new CustomerModel(), 'business_name', $id)],
            'formated_mobile' => ['phone:' . $request['mobile_country'], new BranchUniqueValidator(new CustomerModel(), 'mobile', $id)],
            'formated_telephone' => ['phone:' . $request['telephone_country'], new BranchUniqueValidator(new CustomerModel(), 'telephone', $id)],
            'formated_fax' => ['phone:' . $request['fax_country']],
            'email' => ['email', new BranchUniqueValidator(new CustomerModel(), 'email', $id)],
            'nic' => ['max:20', new BranchUniqueValidator(new CustomerModel(), 'nic', $id)],
            'website' => 'url',
            'outstanding_limit' => 'required',
            'rep' => ['required'],
            //   'area' => ['required'],
            'invoice_type' => ['required'],
            'customer_grade' => ['sometimes', 'string'],
        ], $customMessages);

        if (!$validator->passes()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()->all()]);
        }

        try {
            $customer = CustomerModel::storeCustomer($customer_details);
            AddressModel::saveAddresses($business_address_details, 'B', $customer->id, 'C');
            AddressModel::saveAddresses($shipping_address_details, 'S', $customer->id, 'C');

            return response()->json(['status' => 'success', 'msg' => __('common.messages.update_successfully'), 'id' => $customer->id, 'business_name' => $customer->business_name, 'full_name' => $customer->fname . ' ' . $customer->lname]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'msg' => __('common.messages.save_error')]);
        }
    }

    public function destroy(Request $request, $id)
    {
        Permission::checkPermission($request, 'DELETE_CUSTOMER');

        $customer = CustomerModel::find($id);
        if ($customer->delete())
            return response()->json(['status' => 'success', 'msg' => __('common.messages.record_deleted')]);
        else
            return response()->json(['status' => 'error', 'msg' => __('common.errors.can_not_delete_record_used_somewhere')]);
    }


    public function getCustomerList(Request $request)
    {
        //  Permission::checkPermission($request, 'VIEW_CUSTOMERS_LIST');
        $customers = CustomerModel::with('rep')->get();


        $role = Auth::user()->roles()->first();
        $user = Auth::user();
        if ($role->id != 1 && Auth::user()->can(['SHOW_OWN_CUSTOMERS'])) {

            $customers = CustomerModel::with('rep')
                ->where('rep_id', $user->id)
                ->get();
        }

        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $customers = $customers->where('invoice_type', '35');
        }

        $delete_customer_permission = Auth::user()->can(['DELETE_CUSTOMER']);
        $block_customer_permission = Auth::user()->can(['BLOCK_CUSTOMER']);
        $files_customer_permission = Auth::user()->can(['SHOW_ATTECH_FILES']);

        $edit_customer_permission = false;
        if (Auth::user()->can('EDIT_CUSTOMER')) {
            $edit_customer_permission = true;
        }

        return Datatables::of($customers)
            ->addColumn('action', function ($customers) use ($block_customer_permission, $files_customer_permission, $edit_customer_permission, $delete_customer_permission) {
                $actions = '';
                if ($files_customer_permission) {
                    $actions .= ' <a class = "btn btn-success btn-xs" href="' . route('file-attech', $customers->id) . '" data-original-title="" title=""><i class="fa fa-file"></i></a>';

                }

                if ($block_customer_permission) {
                    $text = 'fa-ban';
                    $class = 'block-customer';
                    if ($customers->block_status == 'Blocked') {
                        $text = 'fa-check';
                        $class = 'unblock-customer';
                    }
                    $actions .= ' <a class = "btn btn-warning btn-xs '.$class.'" href="javascript:void(0)" data-id="'.$customers->id.'" data-original-title="" title=""><i class="fa ' . $text . '"></i></a>';
                }


                if ($edit_customer_permission) {
                    $actions .= ' <a class = "btn btn-info btn-xs" href="' . url("/customer/" . $customers->id . "/edit") . '" id="edit_customer" data-original-title="" title=""><i class="fa fa-pencil"></i></a> <button class="btn btn-primary btn-xs" id="customer_history"><i class="fa fa-history" aria-hidden="true"></i></button>';
                }
                if ($delete_customer_permission) {
                    $actions .= '&nbsp;<button  class="delete_customer btn btn-danger btn-xs" data-toggle="tooltip" data-placement="right" title="" data-original-title="Delete " aria-describedby="tooltip934027"><i class="fa fa-trash-o" aria-hidden="true"></i></button>';
                }
                return $actions;
            })
            ->addColumn('fullname', function ($customers) {
                $full_name = '<strong>' . $customers->company_name . '</strong><br>';
                $full_name .= $customers->fullname;
                if ($customers->block_status == 'Blocked') {
                    $full_name .= '<span class="badge badge-danger">Blocked</span> Remarks : ' . $customers->block_remarks;
                }
                return $full_name;
            })
            ->addColumn('city', function ($customers) {
                return $customers->customerPrivateAddress->city->name_en ?? '';
            })
            ->addColumn('rep', function ($customers) {
                return $customers->rep->username;
            })
            ->addColumn('outstanding_amount', function ($customers) {
                $outstanding_amount = CommonUtil::calculateCustomerInvoiceOutstanding($customers->id);

                return Helper::formatPrice($outstanding_amount);
            })
            ->rawColumns(['fullname', 'action'])
            ->make(true);
    }



    public function openBlockModal(Request $request, $id){
        Permission::checkPermission($request, 'BLOCK_CUSTOMER');
        $customer = CustomerModel::find($id);
        return view('xcustomer::block-modal', compact('customer'));
    }
    public function blockCustomer(Request $request, $id)
    {
        Permission::checkPermission($request, 'BLOCK_CUSTOMER');
        
        $customer = CustomerModel::find($id);
        if ($customer) {
            $customer->block_remarks = $request->block_reason;
            $customer->block_status = 'Blocked';
            $customer->save();
    
            return redirect()->route('customer.index')
                             ->with('status', 'Customer blocked successfully.');
        } else {
            return redirect()->route('customer.index')
                             ->with('error', 'Something went wrong.');
        }
    }
    
    public function unblockCustomer(Request $request, $id)
    {
        Permission::checkPermission($request, 'BLOCK_CUSTOMER');
        
        $customer = CustomerModel::find($id);
        if ($customer) {
            $customer->block_status = 'Active';
            $customer->save();
    
            return redirect()->route('customer.index')
                             ->with('status', 'Customer unblocked successfully.');
        } else {
            return redirect()->route('customer.index')
                             ->with('error', 'Something went wrong.');
        }
    }
    

    public function getCustomerDetails(Request $request)
    {

        if ($request['phone'] != NULL)
            $request['formated_mobile'] = $request['phone'];
        if ($request['tel'] != NULL)
            $request['formated_telephone'] = $request['tel'];

        $validator = Validator::make($request->all(), [
            'formated_mobile' => 'phone:' . $request['mobile_country'],
            'formated_telephone' => 'phone:' . $request['telephone_country'],
        ]);

        if (!$validator->passes()) {
            return response()->json(['status' => 'error']);
        }

        //$customer = CustomerModel::Where('id', $request['id'])->orWhere('mobile', $request['phone'])->orWhere('telephone', $request['tel'])->first();
        $date = Carbon::now();
        $customer = CustomerModel::find($request['id']);
        $pending_cheques = ChequeModel::where('customer_id', $request['id'])->where('status', 1)->whereDate('cheque_date', '>=', $date)->sum('payment_amount');
        $return_cheque = ChequeModel::where('customer_id', $request['id'])->where('status', 2)->sum('payment_amount');
        if ($customer != NULL)

            return response()->json(['status' => 'success', 'msg' => __('Customer Details'), 'pending_cheques' => $pending_cheques, 'customer' => $customer, 'return_cheque' => $return_cheque]);
        else
            return response()->json(['status' => 'error']);
    }


    public function getCustomerBalanceDetails(Request $request)
    {
        //        $validator = Validator::make($request->all(), [
        //            'customer_id' => 'required',
        //        ]);

        //        if (!$validator->passes()) {
        //            return response()->json(['status' => 'error', 'errors' => $validator->errors()->all()]);
        //        }
        $customer_id = $request['customer_id'];
        $customer = CustomerModel::where('id', $customer_id)->first();

        return response()->json(['status' => 'success', 'msg' => __('Customer Details'), 'customer' => $customer ?? []]);
    }


    public function getSelectTwoCustomerNameFilter(Request $request)
    {
        $term = $request['term'];
        $customers = [];
        $user = Auth::user();
        // if ($term != '')
        $customers = CustomerModel::where('company_name', 'like', '%' . $term . '%')->get();
        if ($user->username != 'admin' && $user->username != 'user35') {
            if (Auth::user()->can(['SHOW_OWN_CUSTOMERS'])) {

                $customers = $customers->where('rep_id', $user->id);
            }
        }
        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $customers = $customers->where('invoice_type', '35');
        }

        return response()->json($customers);
    }

    public function getSelectTwoCustomerCodeFilter(Request $request)
    {
        $term = $request['term'];

        $customers = [];
        $user = Auth::user();
        // if ($term != '')
        $customers = CustomerModel::where('business_name', 'like', '%' . $term . '%')->get();
        if ($user->username != 'admin' && $user->username != 'user35') {

            if (Auth::user()->can(['SHOW_OWN_CUSTOMERS'])) {
                $customers = $customers->where('rep_id', $user->id);
            }

        }
        if ($user->username == 'user35'|| $user->username == 'USER30@GMAIL.COM' || $user->username == 'USER33@GMAIL.COM') {
            $customers = $customers->where('invoice_type', '35');
        }

        return response()->json($customers);
    }

    public function generateExcel(Request $request)
    {
        $customer = CustomerModel::get()->take(10);

        return Excel::download(new CustomersExport($customer), 'customer.csv');
    }
}
