<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Utils\CommonUtil;
use App\Utils\ReferenceUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Pramix\XCustomer\Models\CustomerModel;
use Pramix\XGeneral\Models\OptionModel;
use Pramix\XInvoice\Models\InvoiceModel;
use Pramix\XInvoice\Models\InvoicePaymentModel;
use Pramix\XPayment\Models\ChequeModel;
use Pramix\XUser\Models\Permission;
use Pramix\XUser\Models\User;
use Carbon\Carbon;

class PaymentCreationController extends Controller
{
    public function create(Request $request)
    {
        Permission::checkPermission($request, 'ADD_PAYMENT');

        $data = [
            'page' => 'payment_creation',
            'customer_id' => $request->customer_id_selected,
            'customer_info' => null,
            'outstanding_invoices' => [],
            'old_payments' => [],
            'roles' => User::role(['REPRESENTATIVE', 'REP'])->get(),
        ];

        if ($request->customer_id_selected) {
            $customer_id = $request->customer_id_selected;
            $data['customer_info'] = CustomerModel::find($customer_id);

            $due_invoice_list = CommonUtil::getOutstandingInvoiceListByCustomer($customer_id);
            $invoices_to_process = InvoiceModel::where('customer_id', $customer_id)
                ->whereIn('id', $due_invoice_list)
                ->where('status', 'I')
                ->select('id', 'invoice_code', 'invoice_date', 'total')
                ->get();

            $invoices_to_process->each(function ($inv) {
                $inv->paid_amount = CommonUtil::calculatePaidAmountInvoice($inv->id);
                $inv->balance = CommonUtil::calculateInvoiceOutstanding($inv->id);
            });

            $data['outstanding_invoices'] = $invoices_to_process;

            //Previous Payments
            $customer_payment_ids = InvoicePaymentModel::where('customer_id', $customer_id)
                ->pluck('id')
                ->toArray();

            $old_payments_query = InvoicePaymentModel::whereIn('parent_id', $customer_payment_ids);

            //Filters
            if ($request->filled('filter_payment_ref_no')) {
                $old_payments_query->where('payment_ref_no', 'like', '%' . $request->filter_payment_ref_no . '%');
            }

            if ($request->filled('filter_remarks')) {
                $old_payments_query->where('payment_remarks', 'like', '%' . $request->filter_remarks . '%');
            }

            if ($request->filled('filter_payment_date')) {
                $old_payments_query->whereDate('payment_date', $request->filter_payment_date);
            }

            if ($request->filled('filter_payment_amount')) {
                $old_payments_query->where('payment_amount', $request->filter_payment_amount);
            }

            if ($request->filled('filter_payment_method')) {
                $old_payments_query->where('payment_method', $request->filter_payment_method);
            }

            if ($request->filled('filter_rep')) {
                $old_payments_query->where('rep_id', $request->filter_rep);
            }

            // Final query
            $old_payments = $old_payments_query
                ->orderBy('id', 'desc')
                ->paginate(3);

            $data['old_payments'] = $old_payments;
        }

        return view('payment.create_payment_new', $data);
    }

    /**
     * Store a newly created payment in storage.
     */
    public function store(Request $request)
    {
        Permission::checkPermission($request, 'ADD_PAYMENT');

        // --- 1. SERVER-SIDE VALIDATION ---
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customer,id',
            'payment_amount' => 'required|numeric|gt:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,cheque,credit',
            'cheque_date' => 'required_if:payment_method,cheque|date',
            'cheque_bank' => 'required_if:payment_method,cheque',
            'selected_invoices' => 'required|array|min:1',
            'selected_invoices.*.id' => 'required|exists:invoice,id',
            'selected_invoices.*.amount' => 'required|numeric|gt:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // --- 2. CUSTOM BUSINESS LOGIC VALIDATION ---
        $customer = CustomerModel::find($request->customer_id);

        if ($request->payment_method == 'credit' && $customer->credit_balance < $request->payment_amount) {
            return redirect()->back()->withErrors(['msg' => 'Payment amount exceeds available customer credit balance.'])->withInput();
        }

        $total_allocated = 0;
        foreach ($request->selected_invoices as $id => $details) {
            $total_allocated += $details['amount'];
            // $invoice_balance = CommonUtil::calculateInvoiceOutstanding($id);
            // if ($details['amount'] > $invoice_balance) {
            //     return redirect()->back()->withErrors(['msg' => 'Amount applied to an invoice cannot exceed its outstanding balance.'])->withInput();
            // }
        }

        if (round($total_allocated, 2) != round($request->payment_amount, 2)) {
            return redirect()->back()->withErrors(['msg' => 'The total amount applied to invoices must exactly match the payment amount.'])->withInput();
        }

        // --- 3. DATABASE TRANSACTION ---
        DB::beginTransaction();
        try {
            $cheque_id = null;

            $payment_details = [
                'payment_method' => $request->payment_method,
                'payment_date' => $request->payment_date,
                'cheque_date' => $request->cheque_date ?? null,
                'cheque_bank' => $request->cheque_bank ?? null,
                'payment_ref_no' => $request->payment_ref_no,
                'payment_remarks' => $request->payment_remarks ?? '',
                'rep' => $request->rep ?? null, // <-- FIXED: Added rep to the details array
            ];

            if ($request->payment_method == 'cheque') {
                $cheque = ChequeModel::saveCheque($request->customer_id, $request->payment_amount, $payment_details);
                $cheque_id = $cheque->id;
            }

            // Generate Parent Payment Code
            $customer_type = $customer->invoice_type;
            $prefix = ($customer_type == 10) ? 'IP' : 'BP';
            $payment_code_number = ReferenceUtil::returnReferenceNumber($customer_type, $prefix);
            $payment_code_int = intval($payment_code_number->no) + 1;
            ReferenceUtil::referenceUpdate($payment_code_int, $prefix);
            $payment_code = $prefix . '-' . str_pad($payment_code_int, 4, '0', STR_PAD_LEFT);

            // Create Parent Payment Record
            $parent_payment = InvoicePaymentModel::create([
                'payment_code' => $payment_code,
                'invoice_id' => 0,
                'payment_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'payment_amount' => $request->payment_amount,
                'payment_ref_no' => $request->payment_ref_no,
                'payment_remarks' => $request->payment_remarks,
                'customer_id' => $request->customer_id,
                'rep_id' => $request->rep ?? null,
                'cheque_id' => $cheque_id,
                // 'cheque_status' => ($request->payment_method == 'cheque') ? 1 : 0,
                // 'status' => ($request->payment_method == 'cheque') ? 1 : 0,
                'cheque_status' => 1,
                'status' => 1,
                'payment_remarks' => $request->payment_remarks,
                'user_type' => $customer_type,
                'cheque_date' => $request->cheque_date ?? null,
                'bank_id' => $request->cheque_date ?? null,
                'bank_id' => ($request->payment_method == 'cheque') && !empty($request->cheque_bank) ? $request->cheque_bank : null,
                'created_by' => Auth::id(),
            ]);

            // Create Child Payment Records and Update Invoices
            foreach ($request->selected_invoices as $invoice_id => $details) {
                InvoiceModel::saveInvoicePayment($invoice_id, $details['amount'], $payment_details, $parent_payment->id, $payment_code);
            }

            if ($request->payment_method == 'credit') {
                $customer->credit_balance -= $request->payment_amount;
                $customer->save();
            }

            DB::commit();
            // return redirect()->route('payment.create_new')->with('success', 'Payment saved successfully!');

            switch ($request->button_type) {
                case 'SAVE':
                    return redirect()->route('payment.index')->with('success', 'Payment saved successfully!');
                case 'NEW':
                    return redirect()->route('payment.create_new')->with('success', 'Payment saved successfully! You can add a new payment.');
                case 'SAME_CUSTOMER':
                    return redirect()->route('payment.create_new', ['customer_id_selected' => $request->customer_id])
                        ->with('success', 'Payment saved successfully! You can add another payment for the same customer.');
                default:
                    return redirect()->route('payment')->with('success', 'Payment saved successfully!');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['msg' => 'An error occurred while saving the payment. Please try again. Error: ' . $e->getMessage()])->withInput();
        }
    }
}
