<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Utils\CommonUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Pramix\XCustomer\Models\CustomerModel;
use Pramix\XGeneral\Models\OptionModel;
use Pramix\XInventory\Models\Inventory;
use Pramix\XInvoice\Models\InvoiceModel;
use Pramix\XInvoice\Models\InvoiceProductsModel;
use Pramix\XInvoice\Models\InvoiceReturnModel;
use Pramix\XInvoice\Models\InvoiceReturnProductModel;
use Pramix\XProduct\Models\ProductsModel;
use Pramix\XUser\Models\Permission;
use Carbon\Carbon;

class InvoiceReturnCreationController extends Controller
{
    public function create(Request $request)
    {
        Permission::checkPermission($request, 'ADD_CREDIT_NOTE');

        $data = [
            'page' => 'invoice_return_creation',
            'customer_id' => $request->customer_id_selected,
            'invoice_id' => $request->invoice_code_selected,
            'products' => [],
            'outstanding_invoices' => [],
            'customer_info' => null,
            'customer_invoices' => [],
        ];

        if ($request->customer_id_selected) {
            $customer_id = $request->customer_id_selected;
            $data['customer_info'] = CustomerModel::find($customer_id);

            $data['customer_invoices'] = InvoiceModel::where('customer_id', $customer_id)
                ->where('invoice_code', '!=', '')
                ->get();

            if ($request->invoice_code_selected) {
                $invoice = InvoiceModel::find($request->invoice_code_selected);

                if ($invoice) {
                    $data['products'] = DB::table('invoice_products')
                        ->join('product', 'invoice_products.product_id', '=', 'product.id')
                        ->where('invoice_products.invoice_id', $invoice->id)
                        ->whereNull('invoice_products.deleted_at')
                        ->select('product.id', 'product.item_code', 'product.stock_id')
                        ->get();

                    $due_invoice_list = CommonUtil::getOutstandingInvoiceListByCustomer($invoice->customer_id);
                    $invoices_to_process = InvoiceModel::where('customer_id', $invoice->customer_id)
                        ->whereIn('id', $due_invoice_list)
                        ->where('status', 'I')
                        ->select('id', 'invoice_code', 'invoice_date', 'total')
                        ->get();

                    $invoices_to_process->each(function ($inv) {
                        $inv->paid_amount = CommonUtil::calculatePaidAmountInvoice($inv->id);
                        $inv->balance = CommonUtil::calculateInvoiceOutstanding($inv->id);
                    });

                    $data['outstanding_invoices'] = $invoices_to_process;
                }
            }
        }

        return view('invoice_return.create_return', $data);
    }

    /**
     * Store a newly created invoice return in storage.
     */
    public function store(Request $request)
    {
        Permission::checkPermission($request, 'ADD_CREDIT_NOTE');

        // --- 1. VALIDATION ---
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customer,id',
            // 'invoice_id' => 'required|exists:invoice,id',
            'return_items' => 'required|array|min:1',
            'return_items.*.product_id' => 'required|exists:product,id',
            'return_items.*.quantity' => 'required|numeric|gt:0',
            'return_items.*.unit_price' => 'required|numeric|min:0',
            'deduct_invoices' => 'nullable|array',
            'deduct_invoices.*.id' => 'required_with:deduct_invoices|exists:invoice,id',
            'deduct_invoices.*.amount' => 'required_with:deduct_invoices|numeric|gt:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // --- Custom Validation Logic ---
        $grand_total = 0;
        foreach ($request->return_items as $item) {
            $discountAmount = 0;
            if ($item['discount_type'] === 'P') {
                $discountAmount = ($item['quantity'] * $item['unit_price']) * ($item['discount'] / 100);
            } else {
                $discountAmount = $item['discount'];
            }
            $grand_total += ($item['quantity'] * $item['unit_price']) - $discountAmount;
        }

        $total_credit_applied = 0;
        if ($request->has('deduct_invoices')) {
            foreach ($request->deduct_invoices as $deduction) {
                $total_credit_applied += $deduction['amount'];
            }
        }

        if (round($total_credit_applied, 2) != round($grand_total, 2)) {
            return redirect()->back()->withErrors(['msg' => 'The total credit applied must exactly match the grand total of returned items.'])->withInput();
        }

        // --- 2. DATABASE TRANSACTION ---
        DB::beginTransaction();
        try {
            $customer = CustomerModel::find($request->customer_id);
            $user_id = Auth::id();

            // --- Create the Parent Invoice Return record ---
            $parent_return = new InvoiceReturnModel();
            $parent_return->customer_id = $customer->id;
            $parent_return->customer_type = $customer->invoice_type;
            $parent_return->invoice_id = $request->invoice_id;

            //STORE INVOICE IDS
            $returnItems = $request->return_items;
            $invoiceIds = collect($returnItems)->pluck('invoice_id')->unique()->values()->all();
            $parent_return->invoice_ids = $invoiceIds;

            $parent_return->invoice_return_date = Carbon::parse($request->invoice_return_date_created)->format('Y-m-d');
            $parent_return->status = 'A';
            $parent_return->total = $grand_total;
            $parent_return->balance = $grand_total;
            $parent_return->created_by = $user_id;
            $parent_return->remarks = $request->remarks;

            $last_record = InvoiceReturnModel::where('customer_type', $customer->invoice_type)
                ->orderBy('id', 'desc')->where('invoice_return_code', '!=', '')->first();
            $prefix = $customer->invoice_type == 10 ? 'IG' : 'BG';
            $parent_return->invoice_return_code = OptionModel::generateCode($prefix, 4, $last_record->invoice_return_code ?? NULL);
            $parent_return->save();

            // --- Process each returned product ---
            foreach ($request->return_items as $item) {
                $product = ProductsModel::find($item['product_id']);

                InvoiceReturnProductModel::create([
                    'invoice_return_id' => $parent_return->id,
                    'product_id' => $item['product_id'],
                    'description' => $item['description'],
                    'qty' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount' => $item['discount'],
                    'discount_type' => $item['discount_type'],
                    'sub_total' => $item['sub_total'],
                    'discarded' => $item['discarded'],
                    'status' => 1,
                    'created_by' => $user_id,
                ]);

                if ($product->type == 'stock' && $item['discarded'] == 0) {
                    Inventory::increaseInventory($item['product_id'], getConfigArrayValueByKey('STOCK_TRANSACTION_TYPES', 'invoice_return'), $parent_return->invoice_return_code, $item['quantity'], NULL);
                }
            }

            // --- Create child deduction records ---
            if ($request->has('deduct_invoices')) {
                foreach ($request->deduct_invoices as $deduction) {
                    InvoiceReturnModel::create([
                        'parent_id' => $parent_return->id,
                        'deduct_invoice_id' => $deduction['id'],
                        'invoice_return_code' => $parent_return->invoice_return_code,
                        'customer_id' => $customer->id,
                        'customer_type' => $customer->invoice_type,
                        'invoice_id' => $request->invoice_id,
                        'invoice_return_date' => $parent_return->invoice_return_date,
                        'status' => 'A',
                        'total' => $deduction['amount'],
                        'balance' => $deduction['amount'],
                        'created_by' => $user_id,
                        'remarks' => $deduction['remarks'],
                    ]);
                }
            }

            DB::commit();
            // UPDATED: Redirect back to the create page with a success message
            return redirect()->route('invoice_return.index')->with('success', 'Credit Note created successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['msg' => 'An error occurred while saving. Please try again. Error: ' . $e->getMessage()])->withInput();
        }
    }

    public function getCustomerHistory(Request $request, $customerId)
    {
        Permission::checkPermission($request, 'VIEW_INVOICE');

        $invoices = InvoiceModel::where('customer_id', $customerId)
            ->orderBy('invoice_date', 'desc')
            ->paginate(10);

        $invoices->each(function ($invoice) {
            $invoice->paid_amount_calculated = CommonUtil::calculatePaidAmountInvoice($invoice->id);
            $invoice->balance_calculated = CommonUtil::calculateInvoiceOutstanding($invoice->id);
        });

        if ($request->ajax()) {
            return view('invoice_return._customer_history_modal_content', compact('invoices'))->render();
        }
        return back();
    }

    public function getInvoiceProductDetails(Request $request)
    {
        Permission::checkPermission($request, 'MANAGE_CREDIT_NOTE');

        if ($request['invoice_id'] != '' && $request['product_id'] != '') {
            $invoice_return_product_detail = InvoiceProductsModel::where('invoice_id', $request['invoice_id'])
                ->where('product_id', $request['product_id'])
                ->with('product')
                ->first();

            return response()->json(['status' => 'success', 'invoice_return_product_detail' => $invoice_return_product_detail]);
        }
        return response()->json(['status' => 'error', 'msg' => 'Invalid request']);
    }

    public function getCustomerOutstandingInvoices(Request $request)
    {
        $customerId = $request->customer_id;

        if (!$customerId) {
            return response()->json(['status' => 'error', 'message' => 'No customer selected']);
        }

        $due_invoice_list = CommonUtil::getOutstandingInvoiceListByCustomer($customerId);
        $invoices_to_process = InvoiceModel::where('customer_id', $customerId)
            ->whereIn('id', $due_invoice_list)
            ->where('status', 'I')
            ->select('id', 'invoice_code', 'invoice_date', 'total')
            ->get();

        $invoices_to_process->each(function ($inv) {
            $inv->paid_amount = CommonUtil::calculatePaidAmountInvoice($inv->id);
            $inv->balance = CommonUtil::calculateInvoiceOutstanding($inv->id);
        });

        return response()->json([
            'status' => 'success',
            'data' => $invoices_to_process
        ]);
    }

    public function getInvoiceProducts($invoiceId)
    {
        $products = DB::table('invoice_products')
            ->join('product', 'invoice_products.product_id', '=', 'product.id')
            ->where('invoice_products.invoice_id', $invoiceId)
            ->whereNull('invoice_products.deleted_at')
            ->select(
                'product.id',
                'product.item_code',
                'product.stock_id',
            )
            ->get();

        return response()->json($products);
    }
}
