<?php

namespace App\Http\Controllers;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Pramix\XUser\Models\Permission;

class BackUpController extends Controller
{

    public function index()
    {
        Permission::checkPermission(request(), 'MANAGE_DB_BACKUP');
        $disk = Storage::disk(config('backup.backup.destination.disks')[0]);

        $files = $disk->files(config('backup.backup.name'));

        $backups = [];
        // make an array of backup files, with their filesize and creation date
        foreach ($files as $k => $f) {
            // only take the zip files into account
            if (substr($f, -4) == '.zip' && $disk->exists($f)) {
                $backups[] = [
                    'file_path' => $f,
                    'file_name' => str_replace(config('backup.backup.name') . '/', '', $f),
                    'file_size' => $disk->size($f),
                    'last_modified' => $disk->lastModified($f),
                ];
            }
        }
        // reverse the backups, so the newest one would be on top
        $backups = array_reverse($backups);

        // $cron_job_command = $this->commonUtil->getCronJobCommand();
        // $backup_clean_cron_job_command = $this->commonUtil->getBackupCleanCronJobCommand();
        return view('backup.index', compact('backups'));
    }

    public function create()
    {
        Permission::checkPermission(request(), 'CREATE_DB_BACKUP');
        try {
            // start the backup process
            Artisan::call('backup:run');
            $output = Artisan::output();

            // log the results
            Log::info("Backpack\BackupManager -- new backup started from admin interface \r\n" . $output);

            $output = [
                'status' => 'success',
                'msg' => __('lang_v1.success')
            ];
        } catch (Exception $e) {
            $output = [
                'status' => 'error',
                'msg' => $e->getMessage()
            ];
        }

        return back()->with('status', $output);
    }

    public function download($file_name)
    {
        Permission::checkPermission(request(), 'DOWNLOAD_DB_BACKUP');

        $file = config('backup.backup.name') . '/' . $file_name;
        $disk = Storage::disk(config('backup.backup.destination.disks')[0]);
        if ($disk->exists($file)) {
            $fs = Storage::disk(config('backup.backup.destination.disks')[0])->getDriver();
            $stream = $fs->readStream($file);
            return \Response::stream(function () use ($stream) {
                fpassthru($stream);
            }, 200, [
                "Content-Type" => $fs->getMimetype($file),
                "Content-Length" => $fs->getSize($file),
                "Content-disposition" => "attachment; filename=\"" . basename($file) . "\"",
            ]);
        } else {
            abort(404, "The backup file doesn't exist.");
        }
    }

    public function delete($file_name)
    {
        Permission::checkPermission(request(), 'DELETE_DB_BACKUP');
        $disk = Storage::disk(config('backup.backup.destination.disks')[0]);
        if ($disk->exists(config('backup.backup.name') . '/' . $file_name)) {
            $disk->delete(config('backup.backup.name') . '/' . $file_name);
            return redirect()->back();
        } else {
            abort(404, "The backup file doesn't exist.");
        }
    }
}
